(*  ==== PERVASIVE LIBRARY UTILITIES ====
 *               SIGNATURE
 *
 *  Copyright 2013 Ravenbrook Limited <http://www.ravenbrook.com/>.
 *  All rights reserved.
 *  
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are
 *  met:
 *  
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 *  IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 *  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *  PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Description
 *  -----------
 *  This module performs various operations on lambda expressions concerned
 *  with references to the pervasive library.  The library is accessed in
 *  the same way as any other structure, by SELECT operations on the library
 *  structure.  However, not all pervasives are defined in the library
 *  (integer addition is not), and not all pervasives are referenced
 *  explicitly (exceptions generated by integer addition).  This module
 *  provides tools for sorting these out and providing mappings from
 *  pervasives to fields in the pervasive library.
 *
 *  Diagnostic output levels (see ../utils/diagnostic.sml)
 *  ------------------------
 *    0  none
 *    1  external references when building environment
 *    3  externalised lambda expressions
 *
 *  Revision Log
 *  ------------
 *  $Log: library.sml,v $
 *  Revision 1.14  1997/01/24 14:32:28  matthew
 *  Adding Options
 *
 * Revision 1.13  1996/12/04  16:54:51  matthew
 * Simplifying
 *
 * Revision 1.12  1996/02/23  17:21:56  jont
 * newmap becomes map, NEWMAP becomes MAP
 *
 * Revision 1.11  1995/02/13  13:10:39  matthew
 * Changes to DebuggerTypes.
 *
 *  Revision 1.10  1993/03/10  17:06:08  matthew
 *  Signature revisions
 *
 *  Revision 1.9  1992/08/26  13:45:29  jont
 *  Removed some redundant structures and sharing
 *
 *  Revision 1.8  1992/08/19  18:18:13  davidt
 *  Now uses NewMap instead of Map.
 *
 *  Revision 1.7  1992/05/06  11:45:45  jont
 *  Modified to use augmented lambda calculus
 *
 *  Revision 1.6  1991/11/28  16:32:01  richard
 *  Completely reimplemented this module, tidied it up, and documented it.
 *  The functions provided have changed quite a bit.
 *
 *  Revision 1.5  91/11/12  16:40:54  jont
 *  Removed is_inline from signature, now in MachSpec
 *  
 *  Revision 1.4  91/10/29  14:54:41  jont
 *  Changed interface to get_env_and_bindings to return inverse map of
 *  prims to lambdas to spot calls to prims in order to allocate args
 *  on stack
 *  
 *  Revision 1.3  91/09/24  13:09:43  davida
 *  Added is_inline to signature, for 
 *  optimiser.
 *  
 *  Revision 1.2  91/09/23  12:30:56  jont
 *  Added stuff to deal with implicit exceptions (eg from ADDV)
 *  
 *  Revision 1.1  91/09/05  11:00:10  jont
 *  Initial revision
 *)


require "../utils/map";
require "../lambda/auglambda";

signature LIBRARY = sig
  structure AugLambda   : AUGLAMBDA
  structure NewMap : MAP
  type CompilerOptions

  (*  === BUILD EXTERNAL ENVIRONMENT ===
   *
   *  Examines a lambda expression to see what external references it makes,
   *  then generates an new enclosing lambda expression of the form:
   *
   *    LET
   *      Library = BUILTIN _load_string "pervasive_library"
   *      L1      = SELECT (x, Library)
   *      L2      = SELECT (y, Library)
   *              :
   *    IN
   *      exp
   *
   *  such that L1, L2, etc. are the external objects which are the
   *  pervasives, and exp is the original lambda expression with external
   *  references replaced by references to the lambda variables.  Also
   *  returned is a NewMap.map from pervasives to their corresponding lambda
   *  variables, and a set of the lambda variables generated.
   *)

  val build_external_environment :
    CompilerOptions * AugLambda.LambdaTypes.LambdaExp ->
    (AugLambda.LambdaTypes.Primitive, AugLambda.LambdaTypes.LVar) NewMap.map *
    AugLambda.LambdaTypes.LambdaExp



  (*  === EXTRACT IMPLICIT EXTERNAL REFERENCES ===
   *
   *  Extracts the set of external pervasives referenced implicitly by a
   *  lambda expression.  These are the pervasives such as exceptions
   *  generated by in line pervasives, for example, exception Sum generated
   *  by integer addition.  Sum is external even though integer addition is
   *  not.
   *)

  val implicit_external_references :
    AugLambda.AugLambdaExp -> 
    AugLambda.LambdaTypes.Primitive AugLambda.LambdaTypes.Set.Set


end
